from .xil_fixture import XilFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .config import *
from .helper import *


@metadata(
    tc_id="tca_ecu_port_capture",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaXilEcuPortCapture(XilFixture):
    def setUp(self):

        self.logger.info("tca_ecu_port_capture starting")

    def tearDown(self):

        self.logger.info("tca_ecu_port_capture stopping")

    def test_ecu_port_capture(self):
        """ Test capturing characteristic variable values at runtime 
        using the Signal Capture class.

        This test uses a list of variables and an event channel defined in the A2L file. 
        It connects to the ECU, creates a capture instance based on the event channel
        and applies the necessary configuration, such as timestamping mode and capture type.
        A characteristic variable is then selected from the list, a value is written to it
        and the capture is started along with DAQ measurement using the port associated 
        with the created device.
        The next step is to extract timestamp list, variable value dict and event dict 
        for signal group captured.

        Finally, we verify that values captured matches the value written to the Ecu memory, 
        stop capture and measurement and close the connection between the ECU and the Ecu port.
        """
        
        value_to_write = 4
        expected_variable_list = ["channel1","counter","ampl","period","cycleTime"]
        
        # Open the simulated ecu.
        process = subprocess.Popen(get_exe_path())
    
        # Create a test bench object based on specific test bench infos.
        tb_factory = TestbenchFactory()
        tb_info = tb_factory.GetAvailableTestbenches()[0]
        test_bench = tb_factory.CreateVendorSpecificTestbench2(tb_info.VendorName, 
                                                               tb_info.ProductName,
                                                               tb_info.ProductVersion, 
                                                               tb_info.XILVersion)

        # Define a device object and get a list that contains 
        # all variables from the A2L file.
        port,device,variables = get_Device_and_variables_from_A2L(test_bench)
        
        # Verify that variables are correctly extracted from the A2l file.
        self.assertEqual(variables,
                         expected_variable_list,
                         Severity.BLOCKER,
                         "variables are not compatible with the A2L file")
        
        # Establish the connection and start the communication between 
        # the ECU and the device created by XIL ECU PORT.
        device.StartOnlineCalibration()

        time.sleep(0.5)
        
        # Create a variable reference list to define one or more
        # variable with a physical representation.
        physical_variable_refs = [test_bench.VariableRefFactory.CreateGenericVariableRef(variables[2], 
                                                                                         ValueRepresentation.ePhysicalValue)]
        
        # Task infos represent the event channel in the A2L file used for DAQ measurement.
        task_infos_list = get_task_infos(device)
        
        # Assert that task infos are properly parsed and extracted from the A2L file.
        self.assertEqual(task_infos_list,
                         ['mainLoop'],
                         Severity.BLOCKER,
                         "task infos does not correspond to the A2L file")
        
        # Define a capture_result_writer instance. 
        capturing_factory = test_bench.CapturingFactory
        capture_result_writer = capturing_factory.CreateCaptureResultMemoryWriter() 
        
        # Capture instance is created based on the event channel and 
        # capture memory writer object that stores data written to the ECU.
        # (It represents one of capture types)
        capture = device.CreateCapture(task_infos_list[0], capture_result_writer)
        capture.Variables = physical_variable_refs
        capture.DiscardFetchedData = False
         
        # When the timestamp mode is set to 'relative', timestamp calculation begins 
        # at the moment the capture starts.
        capture.TimestampingMode = TimestampingMode.eRELATIVE
        
        # Assert that the capture instance is configured correctly. 
        self.assertEqual(capture.State,
                         CaptureState.eCONFIGURED,
                         Severity.BLOCKER,
                         "wrong capture configuration!!")
        
        # Start the data logging .
        capture.Start()
     
        # Starts the DAQ measurement of all configured 
        # captures of the device set of this port.
        port.StartMeasurement()
        
        # Define the value that will be written to the Ecu.
        variable_value = test_bench.ValueFactory.CreateFloatScalar(value_to_write)
        
        # Write the variable value in the Ecu memory. 
        device.Write(physical_variable_refs[0], variable_value)
        
        time.sleep(2)
        
        # Assert that the capture is running. 
        self.assertEqual(capture.State,
                         CaptureState.eRUNNING,
                         Severity.BLOCKER,
                         "capture is not running")
        
        # Get a capture object with the method Fetch 
        # and extract timestamp_list, variable values 
        # dict and events dict.
        timestamp_list,references_dict,events_dict = get_capture_result(capture.Fetch(False), physical_variable_refs)

         # Assert that for the second variable(named ampl), 
         # the values list matches the value written to the ECU memory.
        capture_values_condition = all(ref == 4 for ref in references_dict['ampl'])
        self.assertTrue(capture_values_condition,
                        Severity.BLOCKER,
                        "characteristic value changes are not captured")
        
        # Assert that the timestamp of the event SART(as example) is the initial value 
        # in timestamp list.
        # Since the timestamp mode is relative, the round value will be 0.
        self.assertEqual(int(events_dict['eDATAFRAMESTART']),
                         int(float(timestamp_list[0])),
                         Severity.BLOCKER,
                         "mismatech event eDATAFRAMESTART")
        
        # Stop the data logging.
        capture.Stop()
        
        # Stop and close the simulated ecu.
        process.terminate()