import os
import shutil
from mtf.test_utilities.db_generator import DbGenerator


def check_db_file_version(db_gen, filepath):
    """
    Get the version of the DaTaBeX that generated the database file specified in the
    filepath, and compare it with the DaTaBeX version of the MTF installed in Python
    to check if they are identical.

    Args:
        db_gen (DbGenerator): DaTaBeX Generator object
        filepath (str): File path to current DaTaBeX file

    Returns:
        bool: True if the file exists with same version, False if it exists but with a
        different version

    """
    db_filename = os.path.basename(filepath)
    db_file_info = db_gen.get_db_file_info(filepath)
    db_file_version = db_file_info["DaTaBeX_version"]
    db_gen_version = db_gen.get_dbx_version(db_gen.pdbx_cli)
    if db_file_version == db_gen_version:
        print(f"DaTaBeX file {db_filename} is already created with "\
              f"version {db_gen_version}")
        return True
    print(f"DaTaBeX file {db_filename} exists in a different "\
          f"version than {db_gen_version}")
    return False


def generate_db_file():
    """
    Generate a new database file with the DaTaBeX version of the MTF installed in Python
    if the existing file is not created with the same version, or if it does not exist at
    all. If it is created with the same version, no changes will be made.
    """
    db_gen = DbGenerator()
    # Get database folder path and all sub-folders.
    current_folder_path = os.path.join(os.path.dirname(__file__))
    database_path = current_folder_path + "\\config\\Databases"
    sub_folders_db_path = [f.path for f in os.scandir(database_path) if f.is_dir()]

    # Check DaTaBeX folder for each subfolder.
    for folder in sub_folders_db_path:
        updated_db_name = os.path.basename(folder) + ".db"
        updated_db_filepath = database_path + "\\" + updated_db_name
        delete_file_old_version = False

        # If file already exists check its version.
        if os.path.exists(updated_db_filepath):
            delete_file_old_version = not check_db_file_version(db_gen, updated_db_filepath)
            # If file exists with current DaTaBeX version, stop generation.
            if not delete_file_old_version:
                break

        # Generate DaTaBeX file.
        ok, generated_db_filepath = db_gen.get_db_at_runtime(folder+'\\')

        # Check generation.
        if not ok:
            print("Error generating db file from folder:", folder)
            break

        # Move (and delete) generated file from its path to the database folder path.
        shutil.move(generated_db_filepath, database_path)
        shutil.rmtree(os.path.dirname(generated_db_filepath))

        # Delete file if it exists with old DaTaBeX version.
        if delete_file_old_version:
            os.remove(updated_db_filepath)

        # Rename DaTaBeX file with folder name.
        actual_db_name = os.path.basename(generated_db_filepath)
        os.rename(database_path + "\\" + actual_db_name,
                  updated_db_filepath)


generate_db_file()
