from .xil_signal_generator_fixture import XilSignalGeneratorFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .config import *
from .helper import *


@metadata(
    tc_id="tca_generate_can_signal",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaXilSignalGeneration(XilSignalGeneratorFixture):
    def setUp(self):

        self.logger.info("tca_generate_can_signal starting")

    def tearDown(self):

        self.logger.info("tca_generate_can_signal stopping")

    def test_generate_can_signal(self):
        """ Test changing a real CAN signal
        from a DB file using user-defined values.

        This test takes a frame and a signal from a DB file,
        creates a list of user-defined values
        based on the selected signal type and its description,
        defines a signal generator class that takes the real signal path
        and the signal description set as attributes,
        and starts changing the CAN signal using values
        interpolated accordingly.

        Finally, an observer monitors the signal generator
        to verify that the runtime values of the CAN signal
        match those created by the XIL module.
        """
         # Choose a frame from the database to transmit.
        channel = self.com_network.try_get_channel_by_name(CHANNEL_NAME)
        channel.get_frame_by_id(FRAME_ID).start_transmission()

        time.sleep(2)

        # Create a test bench object based on XIL 3.0.0 version.
        tb_factory = TestbenchFactory()
        tb_info = tb_factory.GetAvailableTestbenches()[0]
        test_bench = tb_factory.CreateVendorSpecificTestbench2(tb_info.VendorName, 
                                                               tb_info.ProductName,
                                                               tb_info.ProductVersion, 
                                                               tb_info.XILVersion)
        # Use the ValueFactory
        # to define a list of values for the signal.
        value_factory = test_bench.ValueFactory

        # Define the X vector values.
        signal_times = value_factory.CreateFloatVectorValue(X_VECTOR_VALUES)
        # Define the Y vector values.
        signal_values = value_factory.CreateFloatVectorValue(Y_VECTOR_VALUES)

        signal_value = value_factory.CreateSignalValue(signal_times,
                                                       signal_values)

        # Use SignalFactory
        # to convert the defined list of values
        # into a Signal segment.
        signal_factory = test_bench.SignalFactory
        signal_value_segment = signal_factory.CreateSignalValueSegment()

        # Define the interpolation type to connect different values.
        signal_value_segment.Interpolation = InterpolationTypes.eFORWARD

        # Link the defined signal value to the signal segment.
        signal_value_segment.SignalValue = signal_value

        # Define a description for the Signal segment.
        # This description will be used in the signal description set
        # for signal generator class attribute (lines 103 and 113).
        description_signal = signal_factory.CreateSegmentSignalDescription()
        description_signal.Add(signal_value_segment)
        description_signal.LoopCount = DEFINED_SIGNAL_DESCRIPTION_LOOP_COUNT
        description_signal.Name = DEFINED_SIGNAL_DESCRIPTION_NAME

        # The SIGNAL_VALUE represents the number of additional
        # consecutive signal samples with equidistant timestamps
        # to be created for greater precision.
        # The final Y value list will be generated based on the X vector values
        # and the interpolation type.
        interpolated_signal_values = description_signal.CreateSignalValue(SIGNAL_VALUE)

        # this function displays a plot of the signal
        # that will be used in signal generator part.
        plot_signal(interpolated_signal_values.XVector.Value,
                    interpolated_signal_values.FcnValues.Value,
                    GRAPHIC_TITLE, LABEL)

        # This represents the list of values
        # that will be applied to the CAN signal.
        expected_signal_values = [int(val) for val in interpolated_signal_values.FcnValues.Value]
        sig_desc_set = signal_factory.CreateSignalDescriptionSet()
        sig_desc_set.Add(description_signal)

        # Define the path of the signal.
        variable_ref_factory = test_bench.VariableRefFactory
        signal_ref = variable_ref_factory.CreateGenericVariableRef(SIGNAL_FULL_PATH,
                                                                   ValueRepresentation.eRawValue)

        # Define the signal generator class and its attributes.
        signal_generator_factory = test_bench.SignalGeneratorFactory
        signal_generator = signal_generator_factory.CreateSignalGenerator()
        signal_generator.SignalDescriptionSet = sig_desc_set

        # The Assignments attribute of the signal generator
        # is defined as a dictionary.
        # The key of this dictionary is the path to the CAN signal,
        # defined as a variable reference class.
        # The value of the dictionary is the name that characterizes
        # the signal description of the Signal segment.
        # This creates a link between the signal from the database
        # and the generated signal.
        signal_generator.Assignments = {signal_ref: DEFINED_SIGNAL_DESCRIPTION_NAME}

        # Define an observer to monitor the values of the signal at runtime.
        self.legacy_signal(CHANNEL_NAME,
                               FRAME_ID,
                               SIGNAL_NAME).start_observer(on_change=False)

        # Define and execute all the necessary preconditions
        # to correctly start the signal generator.
        signal_generator.LoadToTarget()

        # Verify that the signal generator is ready after the previous line.
        self.assertEqual(signal_generator.ScriptState,
                         ScriptState.eREADY,
                         Severity.BLOCKER,
                         "Signal generator is not ready to set values")

        # Start changing the signal with the defined values.
        signal_generator.Start()

        # Verify that the signal generator sets values for the CAN signal.
        self.assertEqual(signal_generator.ScriptState,
                         ScriptState.eRUNNING,
                         Severity.BLOCKER,
                         "Signal generator is not working")

        # This instruction is similar to a sleep function for 5 seconds
        # between the start and end of the signal generator process.
        signal_generator.WaitForState([ScriptState.eREADY,
                                       ScriptState.eFINISHED],
                                      SLEEP_TIME)

        # Verify that signal generator has stopped and the process is finished.
        self.assertEqual(signal_generator.ScriptState,
                         ScriptState.eFINISHED,
                         Severity.BLOCKER,
                         "Signal generator doesn't stop")

        # Prepare a list of actual values based on the the result from the observer.
        actual_values = self.legacy_signal(CHANNEL_NAME,
                                              FRAME_ID,
                                               SIGNAL_NAME).stop_observer(True)
        actual_signal_value = []
        for value in actual_values:
            actual_signal_value.append(value[1])

        # Verify that the values obtained by the observer are similar
        # to those created by the signal generator.
        # In both lists, we have user-defined values that are repeated
        # with an occurrence of 3 or 4,
        # based on interpolation between two durations.
        # We verify that this occurrence is consistent
        # for all values selected by the user.
        similarity = all(abs(Counter(actual_signal_value)[val] - Counter(expected_signal_values)[val]) <= 1 for val in Counter(expected_signal_values))
        self.assertTrue(similarity,
                        Severity.BLOCKER,
                        "The can signal is not changed by signal generator values.")