from .xil_ecu_port_fixture import XilEcuPortFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .config import *
from .helper import *


@metadata(
    tc_id="tca_ecu_port_signal_monitor",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaXilEcuPortSignalMonitor(XilEcuPortFixture):
    def setUp(self):

        self.logger.info("tca_ecu_port_signal_monitor starting")

    def tearDown(self):

        self.logger.info("tca_ecu_port_signal_monitor stopping")

    def test_ecu_port_signal_monitor(self):
        """ Test capture characteristic variable values in Runtime
        using signal monitor module.

        This test takes a list of variables and an event channel(task_info) 
        from the A2L file, connect to the Ecu, create a signal monitor 
        instance based on the event channel and a queue size defined by user,
        select a characteristic from the variable list and write a value 
        for this characteristic, start the monitor and collect signal samples 
        until a specific timeout and then extract the timestamp  
        and the variable value from each signal sample into a specific dict.

        Finally, we verify that values obtained by signal monitor match 
        the value written to the Ecu memory and close the connection between the 
        ECU and the Ecu port.
        """
        
        value_to_write = 10.5
   
        # Open the simulated ecu 
        process = subprocess.Popen(get_exe_path())
        
        time.sleep(1)
    
        # Create a test bench object based on specific test bench infos.
        tb_factory = TestbenchFactory()
        tb_info = tb_factory.GetAvailableTestbenches()[0]
        test_bench = tb_factory.CreateVendorSpecificTestbench2(tb_info.VendorName, 
                                                               tb_info.ProductName,
                                                               tb_info.ProductVersion, 
                                                               tb_info.XILVersion)

        # Define a device object and get a list that contains 
        # all variables from the A2L file.
        port,device,variables = get_Device_and_variables_from_A2L(test_bench)
        
        # Verify that variables are correctly extracted from the A2l file.
        self.assertEqual(variables,
                         EXPECTED_VARIABLE_LIST,
                         Severity.BLOCKER,
                         "Variables are not compatible with the A2L file")
        
        # Establish the connection and start the communication between 
        # the ECU and the device created by XIL ECU PORT.
        device.StartOnlineCalibration()

        time.sleep(0.5)
        
        # Create a variable reference list to define one or more
        # variable with a physical representation. 
        physical_variable_refs = [test_bench.VariableRefFactory.CreateGenericVariableRef(variables[2], 
                                                                                         VALUE_REPRESENTATION)]
        
        # Task infos represent the event channel in the A2L file used for DAQ measurement.
        task_infos_list = get_task_infos(device)
        
        # Assert that task infos are properly parsed and extracted from the A2L file.
        self.assertEqual(task_infos_list,
                         ['mainLoop'],
                         Severity.BLOCKER,
                         "task infos does not correspond to the A2L file")
        
        # Signal Monitor is defined with a queue size and 
        # an event channel from the A2L file.
        signal_monitor = device.CreateSignalMonitor(QUEUE_SIZE,task_infos_list[0])
         
        # Assign the variables to the signal monitor for observation. 
        signal_monitor.Signals = physical_variable_refs
        
        # Start the Signal monitor.  
        signal_monitor.Start()
        
        # Assert that the signal monitor is running.
        self.assertEqual(signal_monitor.State,
                         MonitorState.eRUNNING,
                         Severity.BLOCKER,
                         "Signal Monitor is not running")
        
        # Define the value that will be written to the Ecu.
        variable_value = test_bench.ValueFactory.CreateFloatScalar(value_to_write)
        
        # Write the variable value in the Ecu memory.
        device.Write(physical_variable_refs[0], variable_value)
        
        time.sleep(0.5)
        
        # Wait for the monitor to observe the variable values. 
        # Within the timeout period those values will be collected 
        # in a list as a signal Samples object.
        signal_samples_list = signal_monitor.WaitForSignalSample(2)
        
        # Verify that the queue size is the same 
        # as defined in the Monitor class instance. 
        self.assertEqual(signal_monitor.QueueSize,
                         QUEUE_SIZE,
                         Severity.BLOCKER,
                         "Monitor queue size is not respected")
        
        # Stop the signal monitor.
        signal_monitor.Stop()
        
        # Assert that the signal monitor is stopped 
        # after finishing the observation.
        self.assertEqual(signal_monitor.State,
                         MonitorState.eSTOPPED,
                         Severity.BLOCKER,
                         "Monitor is not stopped")
        
        # Get the timestamp and the value from each sample object .
        timestamp_list,reference_dict = get_Monitor_result(signal_samples_list,physical_variable_refs)
        
        # Assert that the timestamp increases consistently 
        # based on the standard requirements.
        monitor_timestamp_condition = all(timestamp_list[i + 1] >= timestamp_list[i] 
                                          for i in range(len(timestamp_list) - 1))
        
        self.assertTrue(monitor_timestamp_condition,
                        Severity.BLOCKER,
                        "The timestamp is not increasing")
        
        # Assert that for each signal sample, the variable value 
        # matches the one written to the ECU memory.
        monitor_values_condition = all(reference_dict[i]['ampl'] == value_to_write 
                                       for i in range(signal_monitor.QueueSize))
        
        self.assertTrue(monitor_values_condition,
                        Severity.BLOCKER,
                        "The characteristic value changes are not captured")
        
        device.StopOnlineCalibration()
        
        port.Disconnect()
        
        # Stop and close the simulated ecu. 
        process.terminate()
        
        time.sleep(1)