from .xil_ecu_port_fixture import XilEcuPortFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .config import *
from .helper import *


@metadata(
    tc_id="tca_ecu_port_read_write_variable",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaXilEcuPortReadWrite(XilEcuPortFixture):
    def setUp(self):

        self.logger.info("tca_ecu_port_read_write_variable starting")

    def tearDown(self):

        self.logger.info("tca_ecu_port_read_write_variable stopping")

    def test_ecu_port_read_write_variable(self):
        """ Test the write and read functionalities of xil ecu port.

        This test takes a list of variables from the A2L file,
        connect to the Ecu, select the first variable from the 
        list, define a variable reference object for this variable 
        with physical representation and write a value in the ecu 
        memory using the variable reference.

        Finally, we read the variable value from the ECU and verify
        that the written value matches the read value
        """
        
        value_to_write = 4711.0
        
        # Open the simulated ecu 
        process = subprocess.Popen(get_exe_path())
        time.sleep(1)

        # Create a test bench object based on specific test bench infos.
        tb_factory = TestbenchFactory()
        tb_info = tb_factory.GetAvailableTestbenches()[0]
        test_bench = tb_factory.CreateVendorSpecificTestbench2(tb_info.VendorName, 
                                                               tb_info.ProductName,
                                                               tb_info.ProductVersion, 
                                                               tb_info.XILVersion)

        # Define a device object and get a list that contains 
        # all variables from the A2L file.
        port,device,variables = get_Device_and_variables_from_A2L(test_bench)
        
        # Verify that variables are correctly extracted from the A2l file.
        self.assertEqual(variables,
                         EXPECTED_VARIABLE_LIST,
                         Severity.BLOCKER,
                         "variables are not compatible with the A2L file")
        
        # Establish the connection and start the communication between 
        # the ECU and the device created by XIL ECU PORT.
        device.StartOnlineCalibration()

        time.sleep(0.5)
        
        # Create a variable reference object to define the variable 
        # with a physical representation.
        var_ref = test_bench.VariableRefFactory.CreateGenericVariableRef(variables[0], 
                                                                         VALUE_REPRESENTATION)

        # Define the value that will be written to the Ecu.
        # The selected variable with physical representation 
        # takes float values.
        variable_value = test_bench.ValueFactory.CreateFloatScalar(value_to_write)

        # Write the variable value in the Ecu memory. 
        device.Write(var_ref, variable_value)

        # Read the variable value from the Ecu memory.
        read_value = device.Read(var_ref)
        
        # Verify that the read value has the same container type as the written value.
        self.assertEqual(read_value.ContainerType,
                         ContainerDataType.eSCALAR,
                         Severity.BLOCKER,
                         "container type is not like defined in the A2L file")

        # Verify that the read value has the same data type as the written value.
        self.assertEqual(read_value.DataType,
                         PrimitiveDataType.eFLOAT,
                         Severity.BLOCKER,
                         "value data type is not like defined in the A2L file")

        # Verify that the read value matches the written value.  
        self.assertEqual(read_value.FloatValue,
                         value_to_write,
                         Severity.BLOCKER,
                         "The value written to the A2L variable differs from the value read from the ECU.")
        
        device.StopOnlineCalibration()
        
        port.Disconnect()
        
        # Stop and close the simulated ecu. 
        process.terminate()
        
        time.sleep(1)