"""Module for helper functions and group-wide utilities"""
import sys
import os
from mtf.libs.mtf_pybinder.ASAM.XIL.Implementation.TestbenchFactory.Testbench import TestbenchFactory
import time
import subprocess
current_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(os.path.join(current_dir, "Python_Enums"))
# Import the python Enum folder for ASAM XIL Ecu port or ASAM XIL Signal generator.
from ASAM.XIL.Interfaces.Testbench.Common.Capturing.Enum.timestampingmode import TimestampingMode
from ASAM.XIL.Interfaces.Testbench.Common.Capturing.Enum.capturestate import CaptureState
from ASAM.XIL.Interfaces.Testbench.Common.Monitoring.Enum.monitorstate import MonitorState
from ASAM.XIL.Interfaces.Testbench.Common.VariableRef.Enum.valuerepresentation import ValueRepresentation
from ASAM.XIL.Interfaces.Testbench.Common.ValueContainer.Enum.containerdatatype import ContainerDataType
from ASAM.XIL.Interfaces.Testbench.Common.ValueContainer.Enum.primitivedatatype import PrimitiveDataType
from ASAM.XIL.Interfaces.Testbench.Common.Script.Enum.scriptstate import ScriptState




def get_A2l_path():
    current_dir = os.path.dirname(os.path.abspath(__file__))  
    return os.path.normpath(os.path.join(current_dir, "..", "..","..", "config", "A2L","XCPlite.a2l"))

def get_exe_path():
    script_dir = os.path.dirname(os.path.abspath(__file__))
    return os.path.join(script_dir, "XCPlite.exe")

def get_Device_and_variables_from_A2L(testbench):
    
    """ This Function creates an instance port for the 
    Ecu port class, then use this port to 
    load the corresponding A2L file of the project, 
    extract the device and the variables from 
    the port configured with the A2L.
    The function also uses the ECU's IP and UDP port, 
    along with the master's (laptop's) IP and UDP port, 
    to configure the UDP connection between the ECU 
    (represented by the device) and the master.
    """
    
    port = testbench.ECUPortFactory.CreateECUPort('EcuPort')

    config = port.LoadConfiguration(get_A2l_path())

    port.Configure(config)
    
    Devices_list = port.Devices
    device = Devices_list[0]
    
    # This instruction is optional, it is used when the Ecu has a dynamic ip address 
    # that is diffrent from what is existing in the A2L file.So the user need to configure 
    # the Ecu port with this specific ip Ecu address.
    # port.ConfigureUDPConnection(device,ip_Ecu_address,Ecu_port,ip_master_address,master_port)
    
    variables_list = device.VariableNames
    
    return port,device,variables_list


def get_task_infos(device):
    
    """Returns the list of event channels(task_info)
    from the A2L file.
    """
    task_infos = device.TaskInfos
    task_names_list = [t.Name for t in task_infos]
    
    return task_names_list

def get_capture_result(result, refs):
    
    """ This function decomposes a capture result object
    and extract with a SignalGroups property a 
    timestamp list , a dictionary of captured values 
    for each variable and a dict of events
    associated with a particular timestamp.
    """
    
    # Get a timestamp list from SignalGroups property    
    list_timestamps = [f'{time_stamp:.5f}' for time_stamp in 
                       result.SignalGroups[0].GetTimestamps(ValueRepresentation.eRawValue).FloatVector]

    reference_dict = {}
    # For each variable selected, retrieve a list of captured values and 
    # store all lists in a reference dictionary using variable names as keys 
    for ref in refs:
        value_list = []
        values = result.SignalGroups[0].GetSignalValues(ref)
        for value in values :
            if value.DataType == PrimitiveDataType.eFLOAT:
              value_list.append(value.FloatValue) 
            elif value.DataType == PrimitiveDataType.eUINT:
              value_list.append(value.UIntValue)
        reference_dict[ref.VariableName] = value_list
            
    event_dict = {}
    
    # Returns a dict of events based on the timestamp like the DATA FRAME START 
    # when timestamp = 0 or STOP for the last value of timestamp list
    for event in result.Events:
        event_dict[event.Type.name] = event.TimeStamp
    
    return list_timestamps,reference_dict,event_dict

def get_Monitor_result(samples_list, variable_refs):
    
    """ This function processes a list of SignalSample objects.  
    For each sample, it extracts the timestamp and the value 
    associated with a specific variable.  
    All timestamps are collected into a single list, and the variable 
    values are grouped into dictionaries keyed by variable name.  
    These individual dictionaries are then combined into a single dictionary 
    that is returned along with the timestamp list.
    """
    
    # Each variable sample has a timestamp and a value 
    timestamp_list = []
    reference_dict = {}

    for i, signal_sample in enumerate(samples_list):
        timestamp_list.append(signal_sample.TimeStamp)

        value_dict = {}
        signal_values = signal_sample.SignalValues

        for j, value in enumerate(signal_values):
            variable_name = variable_refs[j].VariableName
        
            if value.DataType == PrimitiveDataType.eFLOAT:
                value_dict[variable_name] = value.FloatValue
            elif value.DataType == PrimitiveDataType.eUINT:
                value_dict[variable_name] = value.UIntValue
        
        # The reference dict is composed by value dicts with the same length 
        # of monitor queue, each dict contains the variable name as a key and 
        # the variable value as dict value
        reference_dict[i] = value_dict

    return timestamp_list, reference_dict