from .switch_undefined_cycle_fixture import SwitchUndefinedCycleFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .helper import *
from .config import *


@metadata(
    tc_id="tca_try_switch_undefined_cycle_npdu",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaTrySwitchUndefinedCycleNPDU(SwitchUndefinedCycleFixture):
    def setUp(self):
        self.logger.info("tca_try_switch_undefined_cycle_npdu starting")

    # Create the test case core method that holds the main test steps.
    def test_switch_undefined_cycle_npdu(self):
        
        # Select an ecu from the database.
        ecu = self.com_network.try_get_ecu_by_name(ECU_1)
        
        self.assertTrue(ecu,
                        Severity.BLOCKER,
                        "ecu is not defined") 
        
        # Define an Ethernet filter to be used by the listener
        # to capture only NPDU traffic originating from the ECU. 
        npdu_filter = create_npdu_filter() 
        
        # Select an npdu from the ecu.
        npdu_out = ecu.try_get_out_npdu_by_id(NPDU_ID)
        
        # Start npdu transmission with the cycle defined from the DB.
        # Based on the ecu and the pdu id, the original cycle 
        # for this pdu is 100 ms(according to the DB).
        npdu_out.start_transmission()
        sleep(0.5)
        
        # Change the cycle to a user defined value.
        # The specified value must be provided in milliseconds.
        # The repetition parameter defines how many packets will 
        # be transmitted using the incorrect cycle. 
        npdu_out.try_switch_to_undefined_cycle(WRONG_CYCLE,REPETITION)
        
        # Define a listener to capture npdu packets.
        eth_listener = EthBusListener(ETH_RECEIVER_CH, npdu_filter)
        eth_listener.start_listening()
        
        # The sleep duration must take both the repetition 
        # and the incorrect cycle value into account to ensure 
        # the transmission of only packets with the wrong 
        # cycle before reverting to the original cycle. 
        sleep(2)
        
        eth_listener.stop_listening()
        
        # Retrieve all received frames in one list.
        npdus = eth_listener.get_queue()
        
        self.assertNotEqual(len(npdus),
                            0,
                            Severity.BLOCKER,
                            "Queue is empty")
        
        list_timestamp = []
        for npdu in npdus:
            list_timestamp.append(npdu['timestamp'])
         
        # Verify that the original cycle is respected
        # and not the modified one.
        for i in range(1,len(list_timestamp)):
            
            # The listener timestamp is returned in nanoseconds.
            # To unify the time units between timestamp and cycle, we convert
            # also the timestamp to milliseconds by dividing it by 1E6. 
            delta=(list_timestamp[i]-list_timestamp[i-1])/1E6
            
            # The difference should be within the average range between the 
            # min cycle and max cycle provided also in milliseconds(can be found in the DB)
            self.assertTrue(390 <= int(delta) <= 410,
                            Severity.BLOCKER,
                            "New cycle is not respected")
        
        # Reset the listener to properly clean up and flush the queue.
        eth_listener.reset()
        
        # Cancel the use of the modified transmission cycle.
        npdu_out.cancel_undefined_cycle()
        
        # Start again the listener.
        eth_listener.start_listening()
        sleep(2)
        
        eth_listener.stop_listening()
        
        # Retrieve all received frames in new list.
        npdus = eth_listener.get_queue()
        
        self.assertNotEqual(len(npdus),
                            0,
                            Severity.BLOCKER,
                            "Queue is empty")
        
        list_timestamp = []
        for npdu in npdus:
            list_timestamp.append(npdu['timestamp'])
         
        # Verify that the original cycle is respected
        # and not the modified one.
        for i in range(1,len(list_timestamp)):
            
            # To unify the time units between timestamp and cycle, we convert
            # also the timestamp in milliseconds by dividing it by 1E6.
            delta=(list_timestamp[i]-list_timestamp[i-1])/1E6
            
            # The difference should be within the average range between the 
            # min cycle and max cycle provided also in milliseconds.
            self.assertTrue(90 <= int(delta) <= 110,
                            Severity.BLOCKER,
                            "cycle is not returned to old value")
            
        # Stop the transmission.
        npdu_out.stop_transmission()
            
    def tearDown(self):
        self.cleanup_all_controllers()
        # Clean up all network listeners to avoid interference with other tests.
        self.bus_manager.cleanup_all_listeners()
        self.logger.info("tca_try_switch_undefined_cycle_npdu stopping")