from .flexray_fixture import FlexrayFixture
from xtr import Severity
from xtr import metadata, extra_metadata
from .helper import *
from .config import *


@metadata(
    tc_id="tca_capture_flexray_tp_frames",
    description="",
    status="Design",
    authors=["YourName"],
    domain="",
    targets=[("TargetEcu", True)],
    testbench_types=["S", "M", "L"],  # TB Type: S, L ... where I can run this testcase
)
@extra_metadata(
    extra_metadata_1="Some important extra metadata",
)
class TcaCaptureFlexrayTpFrames(FlexrayFixture):
    def setUp(self):
        self.logger.info("tca_capture_flexray_tp_frames starting")

    # Create the test case method that holds the main test steps.
    def test_capture_flexray_tp_frames(self):
        
        # Create two instances of the FlexrayTp class.
        flexray_sender = FlexrayTp()
        
        flexray_receiver = FlexrayTp()
        
        # Create two instances of the FlexrayTpConfig class with the 
        # required parameters. Certain parameter values must be inverted 
        # between the sender and receiver (e.g. RX and TX frame IDs or source 
        # and target addresses) to properly establish communication.
        config_sender = FlexrayTpConfig(FR_CHANNEL_2,
                                        RX_FRAME_ID,
                                        SOURCE_ADDRESS,
                                        TX_FRAME_ID,
                                        TARGET_ADDRESS,
                                        PDU_LENGTH,
                                        FLOW_TIMEOUT,
                                        COMMUNICATION_CHANNEL)
        
        config_receiver = FlexrayTpConfig(FR_CHANNEL_1, 
                                          TX_FRAME_ID,
                                          TARGET_ADDRESS,
                                          RX_FRAME_ID,
                                          SOURCE_ADDRESS,
                                          PDU_LENGTH,
                                          FLOW_TIMEOUT,
                                          COMMUNICATION_CHANNEL)
        
        # Configure each FlexRay TP instance by binding it 
        # to the appropriate configuration instance using the 
        # bind() method.
        bind_sender = flexray_sender.bind(config_sender)
        
        self.assertTrue(bind_sender, 
                        Severity.BLOCKER,
                        "FlexRay tp sender is not correctly configured.")
    
        bind_receiver = flexray_receiver.bind(config_receiver)
        
        self.assertTrue(bind_receiver, 
                        Severity.BLOCKER,
                        "FlexRay tp receiver is not correctly configured.")
        
        # Transmit the flexray tp frame with payload length less 
        # than the pdu length.
        flexray_sender.send(FR_PAYLOAD)
        
        sleep(1.5)
        
        # Transmit the flexray tp frame with payload length more
        # than the pdu length.
        flexray_sender.send(EXTENDED_FR_PAYLOAD)
        
        sleep(1.5)

        flexray_tp_queue = flexray_receiver.get_flexray_tp_queue()

        # The FlexrayTP queue returns each message as a dict with the 
        # complete payload (without segmentation). Since there are 
        # two transmissions, the queue will contain two distinct FLEXRAY messages.
        self.assertTrue(len(flexray_tp_queue)==2, 
                        Severity.BLOCKER,
                        "FlexRay tp queue list is empty.")
        
        # Verify that all messages received have the same 
        # frame id and target address, as they originate 
        # from the same communication flow (sender to receiver).
        for flexray_msg in flexray_tp_queue:

            self.assertEqual(flexray_msg["frame_id"],
                             TX_FRAME_ID,
                             Severity.BLOCKER,
                             "Wrong frame id received.")
            
            self.assertEqual(flexray_msg["targert_address"],
                             TARGET_ADDRESS,
                             Severity.BLOCKER,
                             "Wrong target address received.")
            
        # Verify that the first payload received corresponds to 
        # the first transmitted payload.
        self.assertEqual(flexray_tp_queue[0]["payload"],
                         FR_PAYLOAD,
                         Severity.BLOCKER,
                        "Wrong flexray payload received.") 
        
        # Verify that the second payload received corresponds to the 
        # extended data transmitted.
        # The flexray_receiver.recv() method returns only the payload list 
        # from the last dict stored in the fexray_tp_queue.
        self.assertTrue(flexray_tp_queue[1]["payload"] == flexray_receiver.recv() == EXTENDED_FR_PAYLOAD,
                        Severity.BLOCKER,
                        "Wrong extended FlexRay payload received.")
            
        flexray_sender.close()
        
        flexray_receiver.close()

    def tearDown(self):
        self.cleanup_all_controllers()
        # Clean up all network listeners to avoid interference with other tests
        self.bus_manager.cleanup_all_listeners()
        self.logger.info("tca_capture_flexray_tp_frames stopping")